;
; ROM Address is DC00h => MAX SIZE OF CODE IS 4000h (16384Kb), 32x512-BYTES BLOCKS, MAX OFFSET IS 4000h !!!!
;
; ROM SCAN Check Ctrl, Move code to end of memory, Init Int 19h (and Int 18h) and init chip
; Int 19h Check Disk present/ready and set Int 13h
;
; 5.16. Command DISK_SIZE (!!! TRANSLATE FROM CN, DIFFERENT FROM EN ORIGINAL !!!)
; This command is used to obtain the physical capacity of the USB storage device. After successfully initializing the USB storage device, 
; this command can get the USB total capacity of the storage device. CH375 requests an interrupt to the microcontroller after the command
; execution is completed. The microcontroller can read the interrupt status as the command.
; The operational status of the command. If the operation status is USB_INT_SUCCESS, then the data can be obtained by the RD_USB_DATA command.
; It is usually 8 bytes, and the first 4 bytes are composed of the high byte first. The double word data is the total number of sectors 
; of the USB storage device. The last 4 bytes are the double-word data composed of the high byte first is the number of bytes per sector. 
; The result of multiplying the two data is USB storage in bytes.
; The total capacity of the device. If the sector is not 512 bytes, then the SET_PKT_P_SEC command should be executed to set the total number
; of packets per sector.
;
; 5.21. Command DISK_INQUIRY (!!! TRANSLATE FROM CN, DIFFERENT FROM EN ORIGINAL !!!)
; This command queries the characteristics of the USB storage device. CH375 requests interrupt from the microcontroller after the command 
; execution is completed, the microcontroller can read the interrupt status is used as the operation status of the command. If the operating
; status is USB_INT_SUCCESS, it can be determined by the RD_USB_DATA command Get data. The data is usually 36 bytes, including the 
; characteristics of the USB storage device and the identification information of the manufacturer and product. The command generally not 
; needed, except when analyzing new logical units.
;
; Return examples for some USb Flash Drives:
;
; 256Mb : 00 80 02 02   1F 00 00 00   Ascii-Name-Other, may be contains zero bytes inside (First 8 bytes the same as 512Mb)
; 512Mb : 00 80 02 02   1F 00 00 00   Ascii-Name-Other, may be contains zero bytes inside (First 8 bytes the same as 256Mb)
; 256Gb : 00 00 06 02   47 00 00 00   Ascii-Name-Other, may be contains zero bytes inside 
;
;
; INQUIRY Data Format (https://usb.org/sites/default/files/usbmass-ufi10.pdf, https://www.usb.org/sites/default/files/usb_msc_boot_1.0.pdf)
; -------------------------------------------------------------------------
; |   Bit |   7   |   6   |   5   |   4   |   3   |   2   |   1   |   0   |
; | Byte  |       |       |       |       |       |       |       |       |
; |-------|---------------------------------------------------------------|
; |  0    |       Reserved        |         Peripheral Device Type        |
; |-------|---------------------------------------------------------------|
; |  1    |  RMB  |                        Reserved                       |
; |-------|---------------------------------------------------------------|
; |  2    |  ISO Version  |      ECMA Version     |   ANSI Version (00h)  |
; |-------|---------------------------------------------------------------|
; |  3    |           Reserved            |      Response Data Format     |
; |-------|---------------------------------------------------------------|
; |  4    |                     Additional Length (31)                    |
; |-------|---------------------------------------------------------------|
; |  5-   |                           Reserved                            |
; |  7    |                                                               |
; |-------|---------------------------------------------------------------|
; |  8-   |                      Vendor Information                       |
; |  15   |                                                               |
; |-------|---------------------------------------------------------------|
; |  16-  |                    Product Identification                     |
; |  31   |                                                               |
; |-------|---------------------------------------------------------------|
; |  32-  |                    Product Revision Level                     |
; |  35   |                              n.nn                             |
; -------------------------------------------------------------------------
;
; Peripheral Device Type: identifies the device currently connected to the requested logical unit.
;     00h     Direct-access device (floppy) 
;     01h-04h None/Out of scope
;     05h     CD-ROM Device
;     06h     None/Out of scope
;     07h     Optical memory Device (e.g., Non-CD optical disks)
;     08h-0Dh None/Out of scope
;     0Eh     RBC Direct-access device (e.g., UHD Floppy disks)
;     0Fh-1Fh None/Out of scope (no FDD connected to the requested logical unit)
; 
; RMB: Removable Media Bit: this shall be set to one to indicate removable media.
; 
; ISO/ECMA: These fields shall be zero for the UFI device.
; 
; ANSI Version: must contain a zero to comply with this version of the Specification.
; 
; Response Data Format:  a value of 01h shall be used for UFI device
; 
; The Additional Length field shall specify the length in bytes of the parameters. If the Allocation Length of the Command Packet is 
;                       too small to transfer all of the parameters, the Additional Length shall not be adjusted to reflect the 
;                       truncation.  The UFI device shall set this field to 1Fh.
; 
; The Vendor Identification field contains 8 bytes of ASCII data identifying the vendor of the product. 
;                           The data shall be left aligned within this field.
; 
; The Product Identification field contains 16 bytes of ASCII data as defined by the vendor. 
;                            The data shall be left-aligned within this field.
; 
; The Product Revision Level field contains 4 bytes of ASCII data as defined by the vendor. 
;                            The data shall be left-aligned within this field.  
;                            For a UFI device, this field indicates the firmware revision number.
;
;================================================================================================================================
; We must intercept int 19h and work with it because otherwise we not sure that we will be first in boot seguence !
;================================================================================================================================
; Goal of Flash Drive geometry (we have only Total Sectors Number on Flash Drive from USB Get Size Info): 
; 
; Total Head     Num (For Geometry Calculation) = FFh [255] <------------------------------\
; Max   Head     Num (For BIOS int 13h, Fun 08) = FEh [254] (0..FEh = FFh) [0..254 = 255]>-/ (Starting from 0)
; 
; Total Sector   Num (For Geometry Calculation) = 3Fh [ 63] <------------------------------\
; Max   Sector   Num (For BIOS int 13h, Fun 08) = 3Fh [ 63] (1..3Fh = 3Fh) [1...63 =  63]>-/ (Starting from 1)
; 
; Total Cylinder Num (For Geometry Calculation) = Total Sectors Number / (Total Head Num * Total Sector Num) <--\
; Max   Cylinder Num (For BIOS int 13h, Fun 08) = Total Cylinder Num - 1 >--------------------------------------/ (Starting from 0)
;================================================================================================================================
;
; + 3 attempt to initialize with restart on init
; + Check if active partition present before boot from it
; ? Do check if we are already in memory (in ROM-SCAN or in INT19 ???) - NO NEED
; ? PnP in int13/08 (DOS not support PnP) - NO NEED
;
;================================================================================================================================

JUMPS

GEOM_SECT	=	03Fh				; Fixed SPT and HEAD geometry values
GEOM_HEAD	=	0FFh				; Number of cylinder calculate depend it and TotalPhysSecNum

BASE_PORT	=	260h				; Fixed Base Port Value

DelayCount1mkS	=	4				; Fixed value for 1mkS delay

code		segment

	assume cs:code, es:code, ss:code, ds:code

FirstByte	dw	0AA55h
		db	8

		jmp	Begin

;================================================================================================================================

CRC		db	0-022h

ThisDiskNum	db	80h
BiosNumDrives	db	0
ChipType	db	0

TotalPhysSecNum	dd	0				; Total Physical Sectors Num on USB Drive indepent on it logical format

UsbSizeBuf	db	09 dup (0)			;  8 Bytes for USB Device Size and ending zero
UsbInfoBuf	db	37 dup (0)			; 36 Bytes for USB Device Info and ending zero

PresKeySkipStrt	db	13, 10, 'Press CTRL to SKIP start E-DISK or any key to skip waiting 3 sec...', 13, 10, 0
InitWait	db	13, 10, 'Init... Please wait...', 0
NotFoundAbort	db	13, 10, 'USB Device NOT FOUND, E-Disk NOT INSTALLED !!!', 13, 10, 13, 10, 0
NoBootableDisk	db	13, 10, 'No Bootable Disk Found in System. Press Ctrl-Alt-Del to reboot...', 7, 0
BootFromOurDisk	db	13, 10, 'Starting from E-DISK', 13, 10, 13, 10, 0
NewLine		db	13, 10, 0
Start_CH375	db	13, 10, 'USBDisk BIOS Started OK', 7, 13, 10, 13, 10, 0
UsbDevSize	db	13, 10, 'USB Device Size:[', 0
UsbDevInfo	db	        ' Kb], Info:[', 0
UsbDevInfoEnd	db	        ']', 13, 10, 0
TitleStr	db	13, 10, 'CH375 E-DISK HX.S063 x86 USBDisk BIOS (c) W.ch 2005. Mod. Gleb Larionov, v2.7', 13, 10, 0
CallMenuStr	db	13, 10, 'Press any key until 3 sec to call Menu...', 0
ClearStr	db	    13, '                                         ', 13, 0
MenuStr		db	        'USBDisk BIOS Menu:', 13, 10, 13, 10
		db	        '  Enter: Continue Boot from this USB Disk', 13, 10
		db	        '  Space: Boot from classic HDD/FDD, this USB Disk will be secondary', 13, 10
		db	        '  Esc  : Disable this USB Disk and continue normally boot process', 13, 10, 13, 10
		db	        'Your choice [1-3]: ', 0
EnterStr	db	        'Enter', 0
SpaceStr	db	        'Space', 0
EscStr		db	        'Esc', 0

;================================================================================================================================

WriteTTY	proc	near						; Write character in TTY mode

		push	ax
		push	bx

		mov	ah, 0Eh
		xor	bh, bh
		int	10h

		pop	bx
		pop	ax

		ret

WriteTTY	endp

;================================================================================================================================

DwordToDecAlign	proc	near						; Output decimal unsigned double word in DX:AX with right allignment 
									; (width in BP, no allignment if BP = 0)
		push	ax
		push	bx
		push	cx
		push	dx
		push	di
		push	bp
		pushf

		mov	bx, 10
		xor	di, di
@@Again: 
		mov	cx, ax
		mov	ax, dx
		xor	dx, dx
		div	bx
		xchg	ax, cx
		div	bx
		push	dx
		inc	di
		mov	dx, cx
		or	cx, ax
		jnz	@@Again
@@Write:
		cmp	di, bp
		jae	@@SkipAllign
		sub	bp, di
		mov	cx, bp
		mov	al, ' '
@@Allign:
		call	WriteTTY
		loop	@@Allign
@@SkipAllign:
		mov	cx, di
@@DoOut:
		pop	dx
		add	dl, '0'
		mov	al, dl
		call	WriteTTY
		loop	@@DoOut

		popf
		pop	bp
		pop	di
		pop	dx
		pop	cx
		pop	bx
		pop	ax

		ret

DwordToDecAlign	endp

;================================================================================================================================

WordToHex	proc	near						; Output word in AX as HEX w/o allignment

		push	ax
		mov	al, ah
		call	ByteToHex
		pop	ax

ByteToHex:								; Output byte in AL as HEX w/o allignment
		push	ax
		shr	al, 1
		shr	al, 1
		shr	al, 1
		shr	al, 1
		call	HalfByteToHex
		pop	ax
		and	al, 0Fh

HalfByteToHex:
		add	al, 30h
		cmp	al, 39h
		jbe	@@Store
		add	al, 7

@@Store:
		call	WriteTTY

		retn

WordToHex	endp

;================================================================================================================================

PutS		proc	near

		mov	bx, 7
		mov	al, cs:[si]
		inc	si
		or	al, al
		jz	Do_Ret
		call	WriteTTY
		jmp	short PutS
Do_Ret:
		retn

PutS		endp

;================================================================================================================================

Begin:
		pushf
		push	ds
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		mov	bx, ax
		xor	ax, ax
		mov	ds, ax

		mov	si, offset PresKeySkipStrt
		call	PutS

		mov	si, ds:46Ch			; Timer
		mov	dx, si

TimrLoop:
		test	byte ptr ds:417h, 4
		jnz	RetFar

		mov	ah, 01h
		int	16h
		jnz	KeyPress

		mov	si, ds:46Ch			; Timer
		sub	si, dx
		cmp	si, 3*18
		jbe	TimrLoop
		jmp	Contin

KeyPress:
		xor	ah, ah
		int	16h

Contin:
		mov	ax, ds:413h			; Memory Size in Kilobytes
		push	ax				; Push Original Memory Size 0:413
		mov	dx, ax
		sub	dx, (offset LastByte - offset FirstByte) / 1024	+ 1 ; 3
		mov	ds:413h, dx			; New Memory Size in Kilobytes
		shl	ax, 1
		shl	ax, 1
		shl	ax, 1
		shl	ax, 1
		shl	ax, 1
		shl	ax, 1				; *64 =	Segment	Address	of End Of Memory
		sub	ax, ((offset LastByte -	offset FirstByte) / 1024 + 1) SHL 6 ; Substract	segment	count for our code
		push	es				; Push Original ES
		mov	es, ax
		push	cs
		pop	ds
		xor	si, si
		xor	di, di
		mov	cx, (offset LastByte - offset FirstByte) / 2 + 1
		cld
		push	es
		push	ax
		push	bp
		mov	bp, sp
		mov	word ptr [bp+2], offset	RelocStart
		pop	bp
		repe movsw
		nop
		retf

;--------------------------------------------------------------

RelocStart:
		pop	es				; Pop Original ES
		pop	bp				; Pop Original Memory Size 0:413

		xor	dx, dx
		mov	ds, dx

		mov	si, offset InitWait
		call	PutS

;--- Get USb Device Size and Info -----------------------------

		mov	si, 3

NextInit:
		push	si
		call	InitChipAsHost
		call	Delay100mkS
		call	IsDiskAttached
		pop	si
		jnz	GetSize
		mov	al, 05h				; Reset ALL (Must wait 40 mlS)
		call	OutCmd
		call	Delay40mlS
		dec	si
		jnz	NextInit

NoUsbDev:
		mov	si, offset ClearStr
		call	PutS
		mov	si, offset NotFoundAbort
		call	PutS

		mov	ds:413h, bp			; Restore Memory Size in Kilobytes

		mov	si, ds:46Ch			; Timer
		mov	dx, si

TimLoop:
		mov	ah, 01h
		int	16h
		jnz	KeyPressd

		mov	si, ds:46Ch			; Timer
		sub	si, dx
		cmp	si, 3*18
		jbe	TimLoop
		jmp	RetFar

KeyPressd:
		xor	ah, ah
		int	16h
		jmp	RetFar

GetSize:
		mov	si, offset ClearStr
		call	PutS

		mov	al, 53h				; Disk Size Cmd
		call	OutCmd
		call	ReadData			; Read Disk Size
		jz	NoUsbDev
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jne	NoUsbDev
		call	Delay100mkS
       		mov	al, 28h				; Read USB Data	Cmd
       		call	OutCmd
       		call	InpData
       		mov	dl, 08
       		mov	bx, offset UsbSizeBuf

LoopRead08:
       		call	Delay100mkS
       		call	InpData
       		mov	cs:[bx], al			; Store	Data in	Buffer
       		inc	bx
       		dec	dl
       		jnz	LoopRead08

		mov	al, 58h				; Disk Inquiry Cmd
		call	OutCmd
		call	ReadData
		jz	NoUsbDev
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jne	NoUsbDev
		call	Delay100mkS
		mov	al, 28h				; Read USB Data	Cmd
		call	OutCmd
		call	InpData
		mov	dl, 36
		mov	bx, offset UsbInfoBuf

LoopRead36:
		call	Delay100mkS
		call	InpData
		mov	cs:[bx], al			; Store	Data in	Buffer
		inc	bx
		dec	dl
		jnz	LoopRead36

		mov	si, offset UsbInfoBuf + 08
		mov	cx, 36 - 08

NextChar:
		mov	al, cs:[si]
		or	al, al
		jnz	SkipChar
		mov	byte ptr cs:[si], ' '
SkipChar:
		dec	cx
		inc	si
		or	cx, cx
		jnz	NextChar
DoneChar:
		
       		mov	dh, cs:[UsbSizeBuf + 00]
       		mov	dl, cs:[UsbSizeBuf + 01]
       		mov	ah, cs:[UsbSizeBuf + 02]
       		mov	al, cs:[UsbSizeBuf + 03]

		mov	word ptr cs:[TotalPhysSecNum + 0], ax
		mov	word ptr cs:[TotalPhysSecNum + 2], dx

;--------------------------------------------------------------

		inc	byte ptr ds:475h

		mov	ax, offset NewInt18_19
		mov	di, 64h				; Int19h
		mov	[di-4],	ax
		mov	[di-2],	cs			; Set Int18h

		xchg	ax, [di]
		mov	cx, [di+2]
		mov	[di+2],	cs			; Set Int19h and get it orginal vector into CX:AX

		mov	ds:1F0h, ax			; Int 7C = Saved Original Int	19
		mov	ds:1F2h, cx

		push	cs
		pop	ds

		call	InitChipAsHost
		call	Delay100mkS

RetFar:
		pop	di
		pop	si
		pop	dx
		pop	cx
		pop	bx
		pop	ax
		pop	ds
		popf
		retf

;================================================================================================================================

NewInt18_19:
		push	ax				; Reserved place in stack only
		push	bp
		mov	bp, sp
		mov	word ptr [bp+2], 0		; Zero this reserved place
		pop	bp
		pop	ds				; DS = 0
		sti
		mov	cx, ds:1F2h			; Original Int 19h
		mov	ax, ds:1F0h
		push	cx
		push	ax				; Original vector Int 19h in stack

		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		xchg	ax, ds:64h			; Restore Original Int 19h
		xchg	cx, ds:66h			; BUT IT IS POSSIBLE THAT OUR INT19 ALREADY REMEMBED ANY OTHER BIOS EXTENSION MODULE OR APPLICATION

		mov	si, offset TitleStr
		call	PutS

;--- Show USb Device Size and Info ----------------------------

		mov	si, offset UsbDevSize
		call	PutS

		mov	ax, word ptr cs:[TotalPhysSecNum + 0]
		mov	dx, word ptr cs:[TotalPhysSecNum + 2]

		shr	dx, 1
		rcr	ax, 1 

       		push	bp
       		xor	bp, bp
       		call	DwordToDecAlign
       		pop	bp

		mov	si, offset UsbDevInfo
		call	PutS

		mov	si, offset UsbInfoBuf + 08
		call	PutS

		mov	si, offset UsbDevInfoEnd
		call	PutS

;--------------------------------------------------------------

		mov	si, offset CallMenuStr
		call	PutS

		mov	si, ds:46Ch			; Timer
		mov	dx, si

TimerLoop:
		mov	ah, 01h
		int	16h
		mov	al, 1
		jnz	KeyPressed

		mov	si, ds:46Ch			; Timer
		sub	si, dx
		cmp	si, 3*18
		jbe	TimerLoop
		xor	al, al

KeyPressed:
		push	ax
		mov	si, offset ClearStr
		call	PutS
		pop	ax

		or	al, al
		mov	ax, 1C0Dh
		jz	NoMenu
		xor	ah, ah
		int	16h				; Clear previous pressed
		mov	si, offset MenuStr
		call	PutS

KeyAgain:
		xor	ah, ah
		int	16h				; Get Key

		cmp	ax, 1C0Dh
		mov	si, offset EnterStr
		jz	KeyOK
		cmp	ax, 3920h
		mov	si, offset SpaceStr
		jz	KeyOK
		cmp	ax, 011Bh
		mov	si, offset EscStr
		jz	KeyOK
		jmp	KeyAgain

KeyOK:
		push	ax
		call	PutS
		mov	si, offset NewLine
		call	PutS
		mov	si, offset NewLine
		call	PutS
		pop	ax

NoMenu:							; AX = 1C0Dh/3910h/011Bh
		cmp	ax, 1C0Dh
		je	Continue
		cmp	ax, 3920h
		je	DoBootAs81
		cmp	ax, 011Bh
		jne	Continue
		dec	byte ptr ds:475h
		jmp	JmpToOldInt19

Continue:
		mov	dx, BASE_PORT
		inc	dx
		inc	dx
		in	al, dx
		and	al, 40h
		jz	DoBootAs81

		mov	al, 0Ah				; Multipurpose Cmd (Get	Max LUN/Read Reg/...)
		call	OutCmd
		mov	al, 20h				; Read CH375 Internal Register 20h
		call	OutCmdParam
		call	InpData
		and	al, 20h				; Bit 5	- Is Disk Online ?
		jnz	DiskOnline

DoBootAs81:
		mov	cs:ThisDiskNum,	81h
		call	SetNew13

		mov	dl, 80h
		call	IsDiskBootable
		jnc	JmpToOldInt19

		mov	si, offset NoBootableDisk
		call	PutS

		sti
		jmp	$

JmpToOldInt19:
		pop	di
		pop	si
		pop	dx
		pop	cx
		pop	bx
		pop	ax

		retf					; JMP TO ORIGINAL Int 19h previously placed in stack

DiskOnline:
		pop	di
		pop	si
		pop	dx
		pop	cx
		pop	bx
		pop	ax
		call	SetNew13
		call	Delay100mkS
		xor	ax, ax
		mov	ds, ax
		mov	ss, ax
		mov	sp, 7C00h
		sti
		mov	dl, cs:ThisDiskNum
		call	IsDiskBootable
		jnc	BootFromItMsg

		mov	cs:ThisDiskNum, 81h
		mov	dl, 80h
		call	IsDiskBootable
		jnc	BootFromIt

		mov	dl, 00h
		call	IsDiskBootable
		jnc	BootFromIt

		mov	si, offset NoBootableDisk
		call	PutS

		sti
		jmp	$

BootFromItMsg:
		mov	si, offset BootFromOurDisk
		push	bx
		call	PutS
		pop	bx
BootFromIt:
		push	es
		push	bx

		retf

;================================================================================================================================

SetNew13	proc	near

		mov	si, 4Ch				; Int 13h
		mov	ax, cs
		cmp	ax, [si+2]
		jz	StartCH375
		mov	al, ds:475h
		mov	cs:BiosNumDrives, al
		mov	dx, BASE_PORT
		inc	dx
		inc	dx
		in	al, dx
		and	al, 40h
		jnz	SetNewInt13

		mov	dl, 80h				; Previously detected Device is missed - ATAPi Removable Device (ThisDiskNum = 0)
		push	es
		call	IsDiskBootable
		pop	es
		jc	SetNewInt13
		mov	cs:ThisDiskNum,	0

SetNewInt13:
		mov	ax, offset NewInt13
		mov	cx, cs
		xchg	ax, [si]
		xchg	cx, [si+2]
		mov	ds:18Ch, ax			; Int 63 - Original Int13
		mov	ds:18Eh, cx

StartCH375:
		mov	si, offset Start_CH375
		call	PutS
		call	InitChipAsHost

		; NO RET !!!

SetNew13	endp

;================================================================================================================================

GetIntClrIntCmd	proc	near

		mov	al, 22h				; Get Interrupt	Status and Clear Int Request
		call	OutCmd
		call	InpData

		retn

GetIntClrIntCmd	endp

;================================================================================================================================

InitChipAsHost	proc	near

		call	Delay10mkS
		call	GetIntClrIntCmd
		mov	al, 15h				; Set USB Mode Cmd
		call	OutCmd
		mov	al, 6				; USB Host Mode, Prosuce SOF package automatically
		call	OutCmdParam
		call	Delay1000mkS
		call	InpData
		call	Delay10mkS
		mov	al, 1				; Get Version Cmd
		call	OutCmd
		mov	cs:ChipType, 1
		call	InpData
		cmp	al, 80h
		jb	SetChipType1
		cmp	al, 0C0h
		jnb	SetChipType1
		cmp	al, 0B5h
		jnb	SetChipType2
		cmp	al, 0A2h
		jnb	Del4Ret

SetChipType1:
		mov	cs:ChipType, 0
		jmp	short Del4Ret

SetChipType2:
		mov	cs:ChipType, 2

Del4Ret:
		call	Delay1000mkS

		retn

InitChipAsHost	endp

;================================================================================================================================

CHS_To_LBA	proc	near				; DX:AX	- LBA Address

		mov	al, GEOM_SECT
		mul	dh
		mov	dl, ch
		mov	dh, cl
		and	cx, 3Fh
		dec	cx
		add	cx, ax
		shr	dh, 1
		shr	dh, 1
		shr	dh, 1
		shr	dh, 1
		shr	dh, 1
		shr	dh, 1
		mov	al, GEOM_SECT
		mov	ah, 0
		mul	dx
		mov	dl, GEOM_HEAD
		mul	dx
		add	ax, cx
		adc	dx, 0

		retn

CHS_To_LBA	endp

;================================================================================================================================

IsDiskBootable	proc	near

		xor	cx, cx
		mov	ax, 201h
		mov	es, cx
		mov	bx, 7C00h
		mov	dh, 0
		inc	cx
		mov	es:[bx+510], cx
		int	13h				; READ SECTORS INTO MEMORY, AL = number of sectors to read, CH = track, CL = sector, DH = head, DL= drive, ES:BX -> buffer to fill
		cmp	word ptr es:[bx+510], 0AA55h
		jnz	NoBootable

		cmp	byte ptr es:[bx+0], 0EBh	; If JMP instruction - This is BOOT Sector, No other check needed
		jz	Bootable
		cmp	byte ptr es:[bx+0], 0E9h	; If JMP instruction - This is BOOT Sector, No other check needed
		jz	Bootable

		test	byte ptr es:[bx+1BEh], 80h	; Does 1st partition active ?
		jnz	Bootable
		test	byte ptr es:[bx+1CEh], 80h	; Does 2nd partition active ?
		jnz	Bootable
		test	byte ptr es:[bx+1DEh], 80h	; Does 3rd partition active ?
		jnz	Bootable
		test	byte ptr es:[bx+1EEh], 80h	; Does 4th partition active ?
		jnz	Bootable

NoBootable:
		stc
		retn

Bootable:
		clc
		retn

IsDiskBootable	endp

;================================================================================================================================

Delay1mkS	proc	near

		push	ax
		mov	ah, DelayCount1mkS
		or	ah, ah
		jz	Delay1mkS_Exit

Delay1mkS_Loop:						; PC/XT	PPI port B
		in	al, 61h
		dec	ah
		jnz	Delay1mkS_Loop

Delay1mkS_Exit:
		pop	ax

		retn

Delay1mkS	endp

;================================================================================================================================

Delay10mkS	proc	near

		push	cx
		mov	cx, 10

DelayNmkS_Loop:
		call	Delay1mkS
		loop	DelayNmkS_Loop
		pop	cx

		retn

Delay10mkS	endp

;================================================================================================================================

Delay100mkS	proc	near

		push	cx
		mov	cx, 100
		jmp	short DelayNmkS_Loop

Delay100mkS	endp

;================================================================================================================================

Delay1000mkS	proc	near

		push	cx
		mov	cx, 10

Delay1000mkSLop:
		call	Delay100mkS
		loop	Delay1000mkSLop
		pop	cx

		retn

Delay1000mkS	endp

;================================================================================================================================

Delay40mlS	proc	near

		push	cx
		mov	cx, 40

Delay40mlSLop:
		call	Delay1000mkS
		loop	Delay40mlSLop
		pop	cx

		retn

Delay40mlS	endp

;================================================================================================================================

OutCmd		proc	near

		push	dx
		mov	ah, cs:ChipType
		mov	dx, BASE_PORT
		inc	dx
		or	ah, ah
		push	ax
		jnz	NoZeroChipType
		call	Delay1mkS
		call	Delay1mkS
		pop	ax
		out	dx, al
		call	Delay1mkS
		call	Delay1mkS
		pop	dx

		retn

NoZeroChipType:
		cmp	ah, 2
		jnb	ChipType2
		call	Delay1mkS
		mov	al, 1
		out	dx, al
		call	Delay1mkS

ChipType2:
		call	Delay1mkS
		call	Delay1mkS
		pop	ax
		out	dx, al
		call	Delay1mkS
		call	Delay1mkS
		pop	dx

		retn

OutCmd		endp

;================================================================================================================================

OutCmdParam	proc	near

		push	dx
		mov	dx, BASE_PORT
		out	dx, al
		call	Delay1mkS
		call	Delay1mkS
		pop	dx

		retn

OutCmdParam	endp

;================================================================================================================================

InpData		proc	near

		push	dx
		mov	dx, BASE_PORT
		call	Delay1mkS
		call	Delay1mkS
		in	al, dx
		pop	dx

		retn

InpData		endp

;================================================================================================================================

InpParam	proc	near

		push	dx
		cmp	cs:ChipType, 2
		mov	dx, BASE_PORT
		jb	ChipTypeLess2
		inc	dx

ChipTypeLess2:
		in	al, dx
		and	al, 80h
		pop	dx

		retn

InpParam	endp

;================================================================================================================================

ReadData	proc	near

		call	Delay1mkS
		call	Delay1mkS
		push	cx
		mov	cx, 4096

Loop4K:
		mov	ah, 100

Loop100:
		call	InpParam
		jz	DataReaded
		dec	ah
		jnz	Loop100
		loop	Loop4K
		mov	al, 17h				; Abort	NAK Cmd
		call	OutCmd
		mov	cx, 2000

Loop2k:
		call	Delay10mkS
		loop	Loop2k

DataReaded:						; Get Interrupt	Status and Clear Int Request Cmd
		mov	al, 22h
		call	OutCmd
		call	InpData
		or	cx, cx
		pop	cx

		retn

ReadData	endp

;================================================================================================================================

IsDiskAttached	proc	near

		mov	al, 0Ah				; Multipurpose Cmd (Get	MAX LUN/Read Reg/...)
		call	OutCmd
		mov	al, 1Bh				; ??? Is Disk Attached (Is Disk	Online is other	Fun)
		call	OutCmdParam
		call	InpData
		and	al, 7Fh
		mov	al, 14h				; USB_INT_SUCCESS
		jnz	DoReturn			; Disk Ready
		call	Delay100mkS
		call	GetIntClrIntCmd
		call	Delay100mkS
		mov	al, 51h				; Disk Init Cmd
		call	OutCmd
		call	ReadData
		jz	DoReturn
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jnz	DoReturn
		call	Delay100mkS
		mov	al, 53h				; Disk Size Cmd
		call	OutCmd
		call	ReadData			; Read Disk Size
		jz	DoReturn
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jz	SkipWait
		mov	ax, 10000

Delay100mS_Loop:
		call	Delay10mkS
		dec	ax
		jnz	Delay100mS_Loop
		mov	al, 53h				; Disk Size Cmd
		call	OutCmd
		call	ReadData			; Read Disk Size

SkipWait:
		call	Delay10mkS
		mov	al, 1
		call	OutCmd
		call	InpData
		test	al, 80h
		jz	DoRet
		mov	al, 59h				; Disk Ready Cmd
		call	OutCmd
		call	ReadData
		jz	GetDskErr
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jz	DoRet

GetDskErr:						; Get Disk Error Cmd
		mov	al, 5Ah
		call	OutCmd
		call	ReadData
		call	Delay100mkS

DoRet:
		mov	ax, 14h				; USB_INT_SUCCESS ?
		cmp	ah, al

DoReturn:
		retn

IsDiskAttached	endp

;================================================================================================================================

AbortNAK	proc	near

		call	Delay100mkS
		mov	al, 17h				; Abort	NAK Cmd
		call	OutCmd
		call	Delay100mkS
		call	GetIntClrIntCmd
		jmp	Delay10mkS

AbortNAK	endp

;================================================================================================================================

NewInt13:
		cmp	dl, cs:ThisDiskNum
		jnz	CallOrigInt13
		cmp	ah, 2
		jz	CHS_Read
		cmp	ah, 3
		jz	CHS_Write
		cmp	ah, 4
		jz	DummyRet_OK
		cmp	ah, 8
		jz	GetDrvParm
		cmp	ah, 1
		jbe	DummyRet_OK
		cmp	ah, 40h
		jb	CalOrgInt13Ret
		cmp	ah, 41h				; LBA BIOS Installation	Check
		jz	LBA_BiosInstChk
		cmp	ah, 42h				; LBA Read
		jz	LBA_Read
		cmp	ah, 43h				; LBA Write
		jz	LBA_Write
		cmp	ah, 44h				; LBA Verify
		jz	DummyRet_OK
		cmp	ax, 572Ah			; Internal GetCS Function
		jz	InternalGetCS
		mov	ah, 1

		stc
		retf	2

;================================================================================================================================

LBA_BiosInstChk:
		mov	bx, 0AA55h
		mov	ah, 21h				; LBA BIOS Version 2.1 (EDD-1.1)
		mov	cx, 1				; Extended Disk	Access Functions (AH=42h-44h,47h,48h) supported
		retf	2

;================================================================================================================================

InternalGetCS:
		mov	cx, cs

;================================================================================================================================

DummyRet_OK:
		xor	ah, ah
		retf	2

;================================================================================================================================

CallOrigInt13:
		cmp	dl, 81h
		jb	CalOrgInt13Ret
		cmp	cs:ThisDiskNum,	80h
		jnz	CalOrgInt13Ret
		dec	dl
		int	63h				; Original Int 13h. Already work with original disk (80h if single)
		pushf
		cmp	dl, 80h
		jb	SkipIncrement
		inc	dl

SkipIncrement:
		popf
		retf	2

;================================================================================================================================

CalOrgInt13Ret:
		int	63h				; Original Int 13h. Already work with original disk (80h if single)
		retf	2

;================================================================================================================================

GetDrvParm:
		push	ds
		push	bx
		push	si

		push	cs
		pop	ds

		call	IsDiskAttached
		jz	RetErr
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jnz	RetErr

		mov	ax, word ptr cs:[TotalPhysSecNum + 0]
		mov	dx, word ptr cs:[TotalPhysSecNum + 2]
		mov	cx, GEOM_SECT * GEOM_HEAD
		div	cx				; AX = Number of Cylinders depend on GEOM_SECT, GEOM_HEAD and TotalPhysSecNum
		and	ax, 3FFh			; Mask 10 bit only for BIOS consideration
		dec	ax				; Max Cylinder number (Counted from 0, need decrement to 1)

		mov	ch, al				; CH = Cylinder	Lo 8 Bits
		mov	cl, ah
		shl	cl, 1
		shl	cl, 1
		shl	cl, 1
		shl	cl, 1
		shl	cl, 1
		shl	cl, 1				; CL = Cylinder	HI 2 Bits
		and	cl, 0C0h
		or	cl, GEOM_SECT			; Max Sect Number (calculated from 1, no need decrement 1)
		mov	dh, GEOM_HEAD - 1		; Max Head Number (from 0 to FE = FF summary)
		mov	dl, cs:BiosNumDrives

		pop	si
		pop	bx
		pop	ds

		xor	ax, ax				; No ERROR
		retf	2

RetErr:
		mov	dh, 0
		mov	cx, 0
		mov	dl, 1
		mov	ah, 1
		pop	si
		pop	bx
		pop	ds

		stc					; Set ERROR
		retf	2

;================================================================================================================================

CHS_Read	proc	near

BufferAddr	= word ptr -8
SecNumToTrans	= word ptr -6
LBA_Addr_Lo	= word ptr -4
LBA_Addr_Hi	= word ptr -2

		push	es
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		push	bp
		mov	bp, sp
		sub	sp, 8
		mov	[bp+SecNumToTrans], ax
		call	CHS_To_LBA			; DX:AX	- LBA Address
		jmp	short DoReadMain

;================================================================================================================================

LBA_Read:
		push	es
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		push	bp
		mov	bp, sp
		sub	sp, 8
		mov	ax, [si+2]			; LBA Number of	Block to Transfer
		mov	[bp+SecNumToTrans], ax
		les	bx, [si+4]			; LBA Transfer Buffer
		mov	ax, [si+8]			; LBA Starting Block Number Lo
		mov	dx, [si+0Ah]			; LBA Starting Block Number Hi

DoReadMain:
		mov	[bp+LBA_Addr_Lo], ax
		mov	[bp+LBA_Addr_Hi], dx
		mov	[bp+BufferAddr], bx
		mov	di, 3				; Retry	Count

DoReadAgain:
		call	IsDiskAttached
		jz	TryRetryLastOpn
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jnz	TryRetryLastOpn
		or	di, 40h				; DI = ReadOpn + Retry Count
		mov	bx, [bp+BufferAddr]
		mov	al, 54h				; Disk Read Cmd
		call	OutCmd
		mov	al, byte ptr [bp+LBA_Addr_Lo]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Lo+1]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Hi]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Hi+1]
		call	OutCmdParam
		mov	al, byte ptr [bp+SecNumToTrans]
		mov	ch, 0
		mov	cl, al
		call	OutCmdParam
		shl	cx, 1
		shl	cx, 1
		shl	cx, 1				; CX = Number of 64K Blocks to Read

LoopRd8xCX:
		call	ReadData
		jz	TryRetryLastOpn
		cmp	al, 1Dh
		jnz	TryRetryLastOpn
		mov	al, 28h				; Read USB Data	Cmd
		call	OutCmd
		call	InpData
		mov	dl, 64

LoopRead64:
		call	InpData
		mov	es:[bx], al			; Store	Data in	User Buffer
		inc	bx
		dec	dl
		jnz	LoopRead64
		mov	al, 55h				; Disk Read GO Cmd
		call	OutCmd
		loop	LoopRd8xCX

RW_OpnPrologRet:
		call	ReadData
		jz	TryRetryLastOpn
		cmp	al, 14h				; USB_INT_SUCCESS ?
		clc
		jnz	TryRetryLastOpn
		jmp	short DoRetf2

;================================================================================================================================

TryRetryLastOpn:
		call	AbortNAK
		mov	al, 1				; Get Version Cmd
		call	OutCmd
		call	InpData
		test	al, 80h
		jz	EarlChipVersion
		mov	al, 5Ah				; Get Disk Error Cmd
		call	OutCmd
		call	ReadData
		jz	Abort_NAK
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jz	EarlChipVersion

Abort_NAK:
		call	AbortNAK
		mov	al, 15h				; Set USB Mode Cmd
		call	OutCmd
		mov	al, 6				; USB Host Mode, Produce SOF Package Automatically
		call	OutCmdParam

EarlChipVersion:
		call	Delay100mkS
		call	GetIntClrIntCmd
		mov	ax, di				; DI = LastOperationCode + Retry Count
		dec	di
		and	di, 0Fh				; Mask Retry Count
		stc
		jz	DoRetf2
		test	al, 40h				; Last Operation was Read ?
		jnz	DoReadAgain
		test	al, 20h				; Last Operation was Write ?
		jnz	DoWriteAgain

DoRetf2:
		mov	sp, bp
		pop	bp
		pop	di
		pop	si
		pop	dx
		pop	cx
		pop	bx
		pop	ax
		mov	ah, 0
		jnb	SkipIncAH
		inc	ah

SkipIncAH:
		pop	es
		retf	2

CHS_Read	endp

;================================================================================================================================

CHS_Write	proc	near

BufferAddr	= word ptr -8
SecNumToTrans	= word ptr -6
LBA_Addr_Lo	= word ptr -4
LBA_Addr_Hi	= word ptr -2

		push	es
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		push	bp
		mov	bp, sp
		sub	sp, 8
		mov	[bp+SecNumToTrans], ax
		call	CHS_To_LBA			; DX:AX	- LBA Address
		jmp	short DoWriteMain

;================================================================================================================================

LBA_Write:
		push	es
		push	ax
		push	bx
		push	cx
		push	dx
		push	si
		push	di
		push	bp
		mov	bp, sp
		sub	sp, 8
		mov	ax, [si+2]			; LBA Number of	Block to Transfer
		mov	[bp+SecNumToTrans], ax
		les	bx, [si+4]			; LBA Transfer Buffer
		mov	ax, [si+8]			; LBA Starting Block Number Lo
		mov	dx, [si+0Ah]			; LBA Starting Block Number Hi

DoWriteMain:
		mov	[bp+LBA_Addr_Lo], ax
		mov	[bp+LBA_Addr_Hi], dx
		mov	[bp+BufferAddr], bx
		mov	di, 3				; Retry	Count

DoWriteAgain:
		call	IsDiskAttached
		jz	TryRetryLastOpn
		cmp	al, 14h				; USB_INT_SUCCESS ?
		jnz	TryRetryLastOpn
		or	di, 20h				; DI = WriteOpn	+ Retry	Count
		mov	bx, [bp+BufferAddr]
		mov	al, 56h				; Disk Write Cmd
		call	OutCmd
		mov	al, byte ptr [bp+LBA_Addr_Lo]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Lo+1]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Hi]
		call	OutCmdParam
		mov	al, byte ptr [bp+LBA_Addr_Hi+1]
		call	OutCmdParam
		mov	al, byte ptr [bp+SecNumToTrans]
		mov	ch, 0
		mov	cl, al
		call	OutCmdParam
		shl	cx, 1
		shl	cx, 1
		shl	cx, 1				; CX = Number of 64K Blocks to Write

LoopWr8xCX:
		call	ReadData
		jz	TryRetryLastOpn
		cmp	al, 1Eh
		jnz	TryRetryLastOpn
		mov	al, 2Bh				; Write	USb Data Cmd
		call	OutCmd
		mov	al, 40h				; Data Length
		call	OutCmdParam
		mov	dl, 64

LoopWrite64:						; 
		mov	al, es:[bx]			; Get Datat From User Buffer
		inc	bx
		call	OutCmdParam
		dec	dl
		jnz	LoopWrite64
		mov	al, 57h				; Disk Write GO	Cmd
		call	OutCmd
		loop	LoopWr8xCX

		jmp	RW_OpnPrologRet

CHS_Write	endp

;================================================================================================================================

LastByte	db	0

Reserved	db	(8192 - (offset $ - offset FirstByte)) dup (0)

code		ends

		end
